package utility;

import java.io.FileReader;

import weka.classifiers.CostMatrix;
import weka.classifiers.Evaluation;
import weka.classifiers.meta.CostSensitiveClassifier;
import weka.classifiers.trees.J48;
import weka.core.Debug.Random;
import weka.core.Instances;
import weka.core.Utils;
import weka.core.converters.ConverterUtils.DataSource;


public class UtilityModel {

	private static final int NB_FOLD = 10;
	private static final long SEED = 2015;

	public static void main(String[] args) throws Exception {
		// Example: java -classpath "lib/weka.jar;bin/" utility.UtilityModel -t data/data_survey_sanitized.arff -C data/cost_utility.cost
		
		
		// Parse options
		String filename_data = Utils.getOption("t", args);
		String filename_cost = Utils.getOption("C", args);

		// Load data
		System.out.println("Loading dataset file: " + filename_data);
		Instances data = DataSource.read(filename_data);
		
		// Set the class to predict
		data.setClass(data.attribute("utility"));

		// Load cost matrix
		CostMatrix matrix = new CostMatrix(new FileReader(filename_cost));

		// Randomize and stratifies data
		Random rand = new Random(SEED);
		data.randomize(rand);
		if (data.classAttribute().isNominal())
			data.stratify(NB_FOLD);

		// Perform cross-validation
		System.out.println("Performing cross-validation:");
		CostSensitiveClassifier cls = new CostSensitiveClassifier();
		cls.setClassifier(new J48());
		cls.setCostMatrix(matrix);
		cls.setMinimizeExpectedCost(true);
		System.out.println("\tclassifier: " + cls.getClass().getName() + " " + Utils.joinOptions(cls.getOptions()));

		Evaluation eval = new Evaluation(data, matrix);
		for (int iFold = 0; iFold < NB_FOLD; iFold++) {
			Instances test = data.testCV(NB_FOLD, iFold);
			Instances train = data.trainCV(NB_FOLD, iFold, rand);
			cls.buildClassifier(train);
			eval.evaluateModel(cls, test);
		}
		System.out.println("Done!");
		System.out.printf("Average cost: %.3f", eval.avgCost());
		System.out.println();

		// Full summary
		/*
		 * System.err.println("=== Setup ===");
		 * System.err.println("Classifier: " + cls.getClass().getName() + " " +
		 * Utils.joinOptions(cls.getOptions()));
		 * System.err.print("Attributes: "); for (int iAttribute = 0; iAttribute
		 * < data.numAttributes() - 1; ++iAttribute)
		 * System.err.print(data.attribute(iAttribute).name() + " ");
		 * System.err.println(); System.err.println("Dataset: " +
		 * data.relationName()); System.err.println();
		 * System.err.println(eval.toSummaryString("=== " + NB_FOLD +
		 * "-fold Cross-validation ===", false));
		 * System.err.println("Average cost" + "\t" + eval.avgCost());
		 * System.err.println(eval.toMatrixString());
		 */
		
		// Build classifier
		//new CostSensitiveClassifier();
		//cls.setClassifier(new J48());
		//cls.setCostMatrix(matrix);
		//cls.setMinimizeExpectedCost(true);
		//cls.buildClassifier(data);
		
		// Predict
		//cls.classifyInstance(...);
	}

}
